<?php
declare(strict_types=1);

require_once __DIR__ . '/lib/respond.php';
require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/controllers/AuthController.php';
require_once __DIR__ . '/controllers/SettingsController.php';
require_once __DIR__ . '/controllers/IncidentsController.php';
require_once __DIR__ . '/controllers/MembersController.php';
require_once __DIR__ . '/controllers/TeamsController.php';
require_once __DIR__ . '/controllers/OrdersController.php';
require_once __DIR__ . '/controllers/TransfersController.php';
require_once __DIR__ . '/controllers/ChatController.php';
require_once __DIR__ . '/controllers/EventsController.php';
require_once __DIR__ . '/controllers/AdminOrgsController.php';
require_once __DIR__ . '/controllers/AdminUsersController.php';
require_once __DIR__ . '/controllers/CapabilitiesController.php';

csrf_check();

$uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?? '/';
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

// tiny router
function route(string $m, string $path, callable $fn): bool {
  global $method, $uri;
  if ($method !== $m) return false;
  if ($uri !== $path) return false;
  $fn();
  return true;
}

function route_pat(string $m, string $pattern, callable $fn): bool {
  global $method, $uri;
  if ($method !== $m) return false;
  if (preg_match($pattern, $uri, $mch)) {
    $fn($mch);
    return true;
  }
  return false;
}

// Events (SSE)
if ($uri === '/events') {
  EventsController::stream();
  exit;
}

// --- Auth
if (route('POST', '/api/login', fn()=>AuthController::login())) return;
if (route('POST', '/api/logout', fn()=>AuthController::logout())) return;
if (route('GET', '/api/me', fn()=>AuthController::me())) return;

// Public settings (for login page)
if (route('GET', '/api/public-settings', fn()=>SettingsController::publicSettings())) return;

// Admin settings
if (route('GET', '/api/settings', fn()=>SettingsController::getAll())) return;
if (route('POST', '/api/settings', fn()=>SettingsController::update())) return;
if (route('POST', '/api/settings/upload', fn()=>SettingsController::uploadAsset())) return;

// Admin orgs / users
if (route('GET', '/api/admin/orgs', fn()=>AdminOrgsController::list())) return;
if (route('POST', '/api/admin/orgs', fn()=>AdminOrgsController::create())) return;
if (route_pat('PUT', '#^/api/admin/orgs/([a-f0-9\-]{36})$#', fn($m)=>AdminOrgsController::update($m[1]))) return;
if (route_pat('DELETE', '#^/api/admin/orgs/([a-f0-9\-]{36})$#', fn($m)=>AdminOrgsController::deactivate($m[1]))) return;

if (route('GET', '/api/admin/users', fn()=>AdminUsersController::list())) return;
if (route('POST', '/api/admin/users', fn()=>AdminUsersController::create())) return;
if (route_pat('PUT', '#^/api/admin/users/(\d+)$#', fn($m)=>AdminUsersController::update((int)$m[1]))) return;
if (route_pat('POST', '#^/api/admin/users/(\d+)/reset-password$#', fn($m)=>AdminUsersController::resetPassword((int)$m[1]))) return;

// Incidents
if (route('GET', '/api/incidents', fn()=>IncidentsController::list())) return;
if (route('POST', '/api/incidents', fn()=>IncidentsController::create())) return;
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})$#', fn($m)=>IncidentsController::get($m[1]))) return;
if (route_pat('PUT', '#^/api/incidents/([a-f0-9\-]{36})$#', fn($m)=>IncidentsController::update($m[1]))) return;

// Members (directory) + check-in/out
if (route('GET', '/api/members', fn()=>MembersController::listDirectory())) return;
if (route('POST', '/api/members', fn()=>MembersController::upsertMember())) return;
if (route_pat('DELETE', '#^/api/members/([a-f0-9\-]{36})$#', fn($m)=>MembersController::deactivate($m[1]))) return;

// Capabilities catalog
if (route('GET', '/api/capabilities', fn()=>CapabilitiesController::list())) return;
if (route('POST', '/api/admin/capabilities', fn()=>CapabilitiesController::create())) return;
if (route_pat('PUT', '#^/api/admin/capabilities/(\d+)$#', fn($m)=>CapabilitiesController::update((int)$m[1]))) return;
if (route_pat('DELETE', '#^/api/admin/capabilities/(\d+)$#', fn($m)=>CapabilitiesController::remove((int)$m[1]))) return;

if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/participants$#', fn($m)=>MembersController::listParticipants($m[1]))) return;
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/roster$#', fn($m)=>MembersController::orgRoster($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/checkin$#', fn($m)=>MembersController::checkIn($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/checkin-quick$#', fn($m)=>MembersController::checkInQuick($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/checkout$#', fn($m)=>MembersController::checkOut($m[1]))) return;

// Teams
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/teams$#', fn($m)=>TeamsController::list($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/teams$#', fn($m)=>TeamsController::create($m[1]))) return;
if (route_pat('PUT', '#^/api/teams/([a-f0-9\-]{36})$#', fn($m)=>TeamsController::update($m[1]))) return;
if (route_pat('POST', '#^/api/teams/([a-f0-9\-]{36})/assign$#', fn($m)=>TeamsController::assignMembers($m[1]))) return;

// Orders
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/orders$#', fn($m)=>OrdersController::list($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/orders$#', fn($m)=>OrdersController::create($m[1]))) return;
if (route_pat('PUT', '#^/api/orders/([a-f0-9\-]{36})$#', fn($m)=>OrdersController::update($m[1]))) return;
if (route_pat('POST', '#^/api/orders/([a-f0-9\-]{36})/note$#', fn($m)=>OrdersController::addNote($m[1]))) return;

// Transfers / loans
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/transfers/order$#', fn($m)=>TransfersController::offerOrder($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/loans/member$#', fn($m)=>TransfersController::offerMemberLoan($m[1]))) return;
if (route('GET', '/api/inbox', fn()=>TransfersController::inbox())) return;
if (route_pat('POST', '#^/api/transfer/([a-f0-9\-]{36})/decide$#', fn($m)=>TransfersController::decide($m[1]))) return;
if (route_pat('POST', '#^/api/transfer/([a-f0-9\-]{36})/cancel$#', fn($m)=>TransfersController::cancel($m[1]))) return;
if (route_pat('POST', '#^/api/loans/([a-f0-9\-]{36})/end$#', fn($m)=>TransfersController::endLoan($m[1]))) return;

// Chat
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/chat$#', fn($m)=>ChatController::list($m[1]))) return;
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/chat/targets$#', fn($m)=>ChatController::targets($m[1]))) return;
if (route_pat('GET', '#^/api/incidents/([a-f0-9\-]{36})/chat/pins$#', fn($m)=>ChatController::pins($m[1]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/chat/([a-f0-9\-]{36})/pin$#', fn($m)=>ChatController::pin($m[1], $m[2]))) return;
if (route_pat('POST', '#^/api/incidents/([a-f0-9\-]{36})/chat$#', fn($m)=>ChatController::send($m[1]))) return;

api_error(404, 'Route not found');
